import React, { useState, useEffect } from 'react';
import { Search, ChevronDown, Plus, Settings, LogIn, LogOut, FileText } from 'lucide-react';

export const Header: React.FC = () => {
  const [currentTime, setCurrentTime] = useState(new Date());
  const [isClockedIn, setIsClockedIn] = useState(false);

  useEffect(() => {
    const timer = setInterval(() => {
      setCurrentTime(new Date());
    }, 1000);
    return () => clearInterval(timer);
  }, []);

  const formatTime = (date: Date) => {
    return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit', second: '2-digit', hour12: true });
  };

  return (
    <header className="flex flex-col shrink-0 z-50">
      {/* Top Global Bar - Frosted Dark/Translucent */}
      <div className="h-12 flex items-center justify-between px-6 text-white glass-effect border-none" style={{ background: 'transparent', backdropFilter: 'blur(20px)' }}>
        <div className="flex items-center gap-8">
          <div className="flex gap-1 h-12">
            {['Leads', 'Deals', 'Inventory', 'Customers', 'Sales', 'Analytics', 'More'].map(item => (
              <div key={item} className={`nav-link cursor-pointer ${item === 'Deals' ? 'nav-link-active' : 'text-white/80 hover:text-white'}`}>
                {item}
                {item === 'Deals' && <span className="bg-danger text-white text-[10px] px-1.5 rounded-sm font-extrabold ml-1.5 shadow-sm">2</span>}
                {['Inventory', 'Customers', 'Sales', 'Analytics', 'More'].includes(item) && <ChevronDown size={14} className="opacity-50 ml-1" />}
              </div>
            ))}
          </div>
        </div>

        <div className="flex items-center gap-6">
          <div className="flex items-center gap-2">
            <button className="bg-white/10 hover:bg-white/20 px-4 py-1.5 rounded-full text-[12px] font-bold transition-all border border-white/10">Helpdesk <span className="text-danger ml-1">8</span></button>
            <button className="bg-white/10 hover:bg-white/20 px-3 py-1.5 rounded-full text-[12px] font-bold transition-all border border-white/10 flex items-center gap-1.5" title="Notes">
              <FileText size={14} className="text-white/70" />
              Note
            </button>
          </div>

          <div className="flex items-center gap-4 bg-white/5 pl-4 pr-1 py-1 rounded-full border border-white/10 ml-2">
            <div className="flex flex-col items-end">
              <div className="flex items-center gap-2">
                <div className={`w-2 h-2 rounded-full shadow-[0_0_8px] transition-all duration-300 ${isClockedIn ? 'bg-red-500 shadow-red-500 animate-pulse' : 'bg-green-500 shadow-green-500'}`}></div>
                <span className="text-[12px] font-bold leading-none">Admin</span>
              </div>
              <span className="text-[10px] text-white/50 font-medium flex items-center gap-1 mt-0.5">
                {formatTime(currentTime)}
              </span>
            </div>

            <div className="h-4 w-[1px] bg-white/10 mx-1"></div>

            <button
              onClick={() => setIsClockedIn(!isClockedIn)}
              className={`flex items-center gap-2 px-3 py-1 rounded-full text-[10px] font-black transition-all border shadow-lg ${isClockedIn
                ? 'bg-red-500/10 text-red-500 border-red-500/30 hover:bg-red-500/20'
                : 'bg-green-500/10 text-green-400 border-green-500/30 hover:bg-green-500/20'
                }`}
            >
              {isClockedIn ? <LogOut size={12} strokeWidth={3} /> : <LogIn size={12} strokeWidth={3} />}
              {isClockedIn ? 'OUT' : 'IN'}
            </button>

            <div className="w-8 h-8 rounded-full bg-gradient-to-tr from-blue-400 to-indigo-500 overflow-hidden border-2 border-white/40 shadow-xl cursor-pointer hover:scale-110 transition-all ml-1">
              <img src="https://img.freepik.com/premium-vector/vector-flat-illustration-grayscale-avatar-user-profile-person-icon-gender-neutral-silhouette-profile-picture-suitable-social-media-profiles-icons-screensavers-as-templatex9xa_719432-2210.jpg?semt=ais_hybrid&w=740&q=80" alt="user" />
            </div>
          </div>
        </div>
      </div>

      {/* CRM Specific Toolbar - Light White Glass */}
      <div className="h-16 flex items-center justify-between px-6 bg-transparent">
        <div className="flex items-center gap-8">
          <div>
            <div className="flex items-center gap-4">
              <h2 className="text-[20px] font-[400] text-white">Deals</h2>
            </div>
          </div>

          <div className="flex items-center ui-btn-primary  overflow-hidden shadow-xl">
            <button className="flex font-[500] items-center gap-2 px-6 py-[7px] text-inherit font-black text-[14px] border-r border-black/10 hover:bg-white/5 transition-all">
              <Plus size={20} strokeWidth={3} />
              CREATE
            </button>
            <button className="px-3 py-2.5 hover:bg-white/10 transition-colors">
              <ChevronDown size={18} />
            </button>
          </div>

          <div className="filter-ui-components flex justify-center">
            <div className="relative w-[400px]">
              {/* Search Icon */}
              <svg
                className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400"
                width="20"
                height="20"
                fill="none"
                stroke="currentColor"
                strokeWidth="2"
                viewBox="0 0 24 24"
              >
                <circle cx="11" cy="11" r="8" />
                <line x1="21" y1="21" x2="16.65" y2="16.65" />
              </svg>

              {/* Input */}
              <input
                type="search"
                placeholder="Search your items..."
                className="
                  w-full
                  bg-white/20
                  backdrop-blur-xl
                  border border-white/30
                  rounded-xl
                  pl-5 pr-4 py-2
                  text-white
                  placeholder-gray-300
                  text-[12px]
                  outline-none
                  transition-all duration-300
                  focus:bg-white/30
                  focus:border-white/60
                  focus:ring-2 focus:ring-white/40
                "
              />
            </div>
          </div>

        </div>

        <div className="flex items-center gap-4">
          <button className="p-1 text-white/80 bg-white/10 hover:bg-white/20 rounded-[3px] border border-white/10 shadow-lg"><Settings size={20} /></button>
        </div>
      </div>
    </header>
  );
};